import type { Metadata } from "next";
import { setRequestLocale } from "next-intl/server";
import { Navbar } from "@/components/navbar";
import {
  Hero,
  Features,
  WhyUs,
  Services,
  Solutions,
  FeaturedProject,
  Portfolio,
  Testimonials,
  IndustryExpertise,
  Partners,
  Blogs,
  ContactUs,
  Footer,
} from "@/modules/landing/components";
import {
  getProjects,
  getServices,
  getPartners,
  getSettings,
  getRecentBlogs,
  mapSettingToContactInfo,
} from "@/core/lib/api";
import { buildMetadataFromSeo } from "@/core/lib/seo";
import VideoHero from "@/modules/landing/components/video-hero";

type Props = {
  params: Promise<{ locale: string }>;
};

export async function generateMetadata({ params }: Props): Promise<Metadata> {
  const { locale } = await params;
  // Home single type holds homepage-level SEO in Strapi
  const homeRes = await fetch(
    `${process.env.NEXT_PUBLIC_STRAPI_URL ?? process.env.STRAPI_URL ?? "http://localhost:1337"}/api/home?populate[seo][populate][metaImage][fields][0]=url&populate[seo][populate][metaImage][fields][1]=alternativeText&populate[seo][populate][openGraph][populate][ogImage][fields][0]=url&populate[seo][populate][openGraph][populate][ogImage][fields][1]=alternativeText&populate[openGraph][populate][ogImage][fields][0]=url&populate[openGraph][populate][ogImage][fields][1]=alternativeText&locale=${locale}`,
    {
      headers: { "Content-Type": "application/json" },
      cache: "no-store",
    }
  );

  let homeSeo: any = null;
  if (homeRes.ok) {
    const json = await homeRes.json();
    homeSeo = json?.data ?? null;
  }

  return buildMetadataFromSeo({
    seo: homeSeo?.seo,
    openGraph: homeSeo?.openGraph,
    locale: (locale === "ar" ? "ar" : "en") as "en" | "ar",
  });
}

export default async function HomePage({ params }: Props) {
  const { locale } = await params;
  setRequestLocale(locale);

  const [allProjects, services, partners, settings, recentBlogs] = await Promise.all([
    getProjects(),
    getServices(),
    getPartners(),
    getSettings(),
    getRecentBlogs(4),
  ]);

  const contactInfo = mapSettingToContactInfo(settings);
  const contactServiceOptions = services.map((s) => ({ name: s.name, documentId: s.documentId }));
  const featuredProjects = allProjects.slice(0, 6);

  return (
    <div className="min-h-screen">
      <Navbar />
      {/* <Hero /> */}
      <VideoHero />
      <Features />
      <WhyUs />
      <Services apiServices={services} />
      <Solutions />
      <FeaturedProject projects={featuredProjects} />
      <Portfolio projects={allProjects} />
      <Testimonials />
      <IndustryExpertise />
      <Partners partners={partners} />
      <Blogs apiBlogs={recentBlogs} />
      <ContactUs contactInfo={contactInfo ?? undefined} serviceOptions={contactServiceOptions} />
      <Footer />
    </div>
  );
}
